/*
 * Decompiled with CFR 0.152.
 */
package cz.insophy.inplan.export.excel;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import cz.insophy.inplan.export.TableWriter;
import cz.insophy.inplan.report.ReportUtilities;
import cz.insophy.inplan.superplan.GeneralizedRequest;
import cz.insophy.inplan.util.Localizer;
import cz.insophy.inplan.util.html.CssColorParser;
import java.awt.Color;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.poi.common.usermodel.HyperlinkType;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.ClientAnchor;
import org.apache.poi.ss.usermodel.Comment;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Drawing;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Hyperlink;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.IndexedColorMap;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFFont;

public class ExcelTableWriter
extends TableWriter {
    public static final int XLS_ROW_LIMIT = 65535;
    private static final int XLSX_ROW_LIMIT = 0x100000;
    private static final int HYPERLINK_LIMIT = 65530;
    private static final int COMMENTS_LIMIT = 10000;
    private static final double MS_PER_HOUR = 3600000.0;
    private static final double MS_PER_DAY = 8.64E7;
    private static final int WRITER_BUFFER = 32768;
    private static final int WIDTH_TRACK_ROWS = 100;
    private final Workbook book;
    private final boolean shouldClose;
    private final OutputStream output;
    private final int rowLimit;
    private Sheet sheet;
    private Drawing<?> drawingPatriarch;
    private Row row;
    private Cell cell;
    private int rowNo;
    private int cellNo;
    private int maxCellNo;
    private boolean autosized;
    private int linksOnSheet;
    private int commentCount;
    private final Map<String, Font> cellColoredFonts = Maps.newHashMap();
    private final CreationHelper creationHelper;
    private final Map<CellStyleInfo, CellStyle> cellStylesCache = Maps.newHashMap();
    private final Font monospaceFont;

    public ExcelTableWriter(File outputFile, boolean useXmlFormat) throws FileNotFoundException {
        this(new BufferedOutputStream(new FileOutputStream(outputFile), 32768), useXmlFormat, true);
    }

    public ExcelTableWriter(OutputStream output, boolean useXmlFormat) {
        this(output, useXmlFormat, false);
    }

    private ExcelTableWriter(OutputStream output, boolean useXmlFormat, boolean shouldClose) {
        this.shouldClose = shouldClose;
        this.output = Preconditions.checkNotNull(output);
        this.book = useXmlFormat ? new SXSSFWorkbook() : new HSSFWorkbook();
        this.rowLimit = useXmlFormat ? 0x100000 : 65535;
        log.info("Excel table writer created with workbook: " + this.book.getClass().getSimpleName());
        this.creationHelper = this.book.getCreationHelper();
        this.initDefaultCellStyles();
        this.sheet = null;
        this.rowNo = 0;
        this.cellNo = 0;
        this.maxCellNo = -1;
        this.row = null;
        this.cell = null;
        this.autosized = true;
        this.monospaceFont = this.createMonospaceFont();
    }

    private Font createMonospaceFont() {
        Font font = this.book.createFont();
        font.setFontName("Courier");
        font.setFontHeightInPoints((short)8);
        return font;
    }

    private CellStyle createCachedCellStyle(CellStyleInfo.Type type, String bkgColor, String txtColor) {
        Font font;
        Color color;
        CellStyle style = this.book.createCellStyle();
        switch (type) {
            case TEXT: {
                break;
            }
            case FLOAT: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("#,##0.##"));
                break;
            }
            case DATETIME: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("d.m.yy h:mm:ss"));
                break;
            }
            case DATE: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("d.m.yy"));
                break;
            }
            case TIME: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("h:mm:ss"));
                break;
            }
            case DURATION_HOUR: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("#,##0\" h\""));
                break;
            }
            case DURATION_DAY: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("#,##0\" d\""));
                break;
            }
            case NORMOHOUR: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("#,##0\" Nh\""));
                break;
            }
            case PERCENT: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("0.00%"));
                break;
            }
            case HEADER: {
                Font headerFont = this.book.createFont();
                headerFont.setBold(true);
                style.setFont(headerFont);
                style.setLocked(true);
                break;
            }
            case HYPERLINK: {
                Font hlink_font = this.book.createFont();
                hlink_font.setFontHeightInPoints((short)10);
                hlink_font.setUnderline((byte)1);
                hlink_font.setColor(IndexedColors.BLUE.getIndex());
                style.setFont(hlink_font);
                break;
            }
            case INTEGER: {
                style.setDataFormat(this.creationHelper.createDataFormat().getFormat("0"));
            }
        }
        if (!Strings.isNullOrEmpty(bkgColor) && style instanceof XSSFCellStyle && (color = CssColorParser.toColor(bkgColor)) != null) {
            style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
            ((XSSFCellStyle)style).setFillForegroundColor(new XSSFColor(color, ((SXSSFWorkbook)this.book).getXSSFWorkbook().getStylesSource().getIndexedColors()));
        }
        if (!Strings.isNullOrEmpty(txtColor) && (font = this.getCellFont(txtColor)) != null) {
            style.setFont(font);
        }
        this.cellStylesCache.put(new CellStyleInfo(type, bkgColor, txtColor), style);
        return style;
    }

    private void initDefaultCellStyles() {
        this.createCachedCellStyle(CellStyleInfo.Type.FLOAT, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.DATETIME, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.DATE, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.TIME, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.DURATION_HOUR, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.DURATION_DAY, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.NORMOHOUR, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.PERCENT, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.HEADER, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.HYPERLINK, null, null);
        this.createCachedCellStyle(CellStyleInfo.Type.INTEGER, null, null);
    }

    private CellStyle getCellStyle(CellStyleInfo.Type type, @Nullable TableWriter.CellProperties props) {
        String txtColor;
        String bckgColor = props == null ? null : props.bckgColor;
        CellStyleInfo csi = new CellStyleInfo(type, bckgColor, txtColor = props == null ? null : props.txtColor);
        CellStyle style = this.cellStylesCache.get(csi);
        if (style != null) {
            return style;
        }
        style = this.createCachedCellStyle(type, bckgColor, txtColor);
        this.cellStylesCache.put(csi, style);
        return style;
    }

    public static String getSafeSheetName(String tableName) {
        Pattern replace = Pattern.compile("[:/\\\\*?\\[\\]]+");
        Matcher matcher = replace.matcher(tableName);
        String res = matcher.replaceAll("_");
        return res.substring(0, Math.min(res.length(), 31));
    }

    @Override
    public void newSheet(String title) {
        this.autosizeCells();
        this.rowNo = 0;
        this.cellNo = 0;
        this.maxCellNo = -1;
        this.row = null;
        this.cell = null;
        this.autosized = false;
        this.linksOnSheet = 0;
        this.sheet = title == null ? this.book.createSheet() : this.book.createSheet(ExcelTableWriter.getSafeSheetName(title));
        if (this.sheet instanceof SXSSFSheet) {
            ((SXSSFSheet)this.sheet).trackAllColumnsForAutoSizing();
        }
        this.drawingPatriarch = this.sheet.createDrawingPatriarch();
    }

    @Override
    public boolean newRow() {
        Preconditions.checkNotNull(this.sheet);
        if (this.rowNo >= this.rowLimit) {
            this.row = null;
            return false;
        }
        if (this.rowNo == 100) {
            this.autosizeCells();
        }
        this.row = this.sheet.createRow(this.rowNo++);
        this.cellNo = 0;
        return true;
    }

    private void newCell(CellStyle style) {
        Preconditions.checkNotNull(this.row);
        this.maxCellNo = Math.max(this.maxCellNo, this.cellNo);
        this.cell = this.row.createCell(this.cellNo++);
        if (style != null) {
            this.cell.setCellStyle(style);
        }
    }

    @Override
    public void newHeaderCell(String title) {
        if (title == null) {
            this.newEmptyCell(null);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.HEADER, null));
            this.cell.setCellValue(title);
        }
    }

    @Override
    public void newStringCell(String value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.TEXT, props));
            this.cell.setCellValue(ReportUtilities.stripHtml(value));
        }
    }

    private Font getCellFont(String color) {
        if (Strings.isNullOrEmpty(color)) {
            return null;
        }
        if (this.cellColoredFonts.containsKey(color)) {
            return this.cellColoredFonts.get(color);
        }
        Color c = CssColorParser.toColor(color);
        if (c != null) {
            Font font = this.book.createFont();
            font.setFontHeightInPoints((short)10);
            Workbook workbook = this.book;
            if (workbook instanceof SXSSFWorkbook) {
                SXSSFWorkbook sxssfWorkbook = (SXSSFWorkbook)workbook;
                IndexedColorMap indexedColors = sxssfWorkbook.getXSSFWorkbook().getStylesSource().getIndexedColors();
                ((XSSFFont)font).setColor(new XSSFColor(c, indexedColors));
            } else {
                HSSFWorkbook hssfWorkbook = (HSSFWorkbook)this.book;
                HSSFColor hssfColor = hssfWorkbook.getCustomPalette().findColor((byte)c.getRed(), (byte)c.getGreen(), (byte)c.getBlue());
                if (hssfColor == null) {
                    hssfColor = hssfWorkbook.getCustomPalette().findSimilarColor((byte)c.getRed(), (byte)c.getGreen(), (byte)c.getBlue());
                }
                font.setColor(hssfColor.getIndex());
            }
            this.cellColoredFonts.put(color, font);
            return font;
        }
        return null;
    }

    @Override
    public void newDateCell(Long date, TableWriter.CellProperties props) {
        this.newDtCell(date, CellStyleInfo.Type.DATE, props);
    }

    @Override
    public void newDatetimeCell(Long date, TableWriter.CellProperties props) {
        this.newDtCell(date, CellStyleInfo.Type.DATETIME, props);
    }

    @Override
    public void newTimeCell(Long date, TableWriter.CellProperties props) {
        this.newDtCell(date, CellStyleInfo.Type.TIME, props);
    }

    private void newDtCell(Long date, CellStyleInfo.Type type, TableWriter.CellProperties props) {
        if (date == null || !GeneralizedRequest.isDateValid(date)) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(type, props));
            this.cell.setCellValue(new Date(date));
        }
    }

    @Override
    public void newDurationInDayCell(Long duration, TableWriter.CellProperties props) {
        this.newDurationCell(duration, CellStyleInfo.Type.DURATION_DAY, props, 8.64E7);
    }

    @Override
    public void newDurationInHourCell(Long duration, TableWriter.CellProperties props) {
        this.newDurationCell(duration, CellStyleInfo.Type.DURATION_HOUR, props, 3600000.0);
    }

    @Override
    public void newDurationCell(Long duration, TableWriter.CellProperties props) {
        this.newDurationInHourCell(duration, props);
    }

    @Override
    public void newBooleanCell(Boolean value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newStringCell(value != false ? Localizer.getString("human_boolean_true") : Localizer.getString("human_boolean_false"), props);
        }
    }

    private void newDurationCell(Long duration, CellStyleInfo.Type type, TableWriter.CellProperties props, double factor) {
        if (duration == null || !GeneralizedRequest.isDateValid(duration)) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(type, props));
            this.cell.setCellValue((double)duration.longValue() / factor);
        }
    }

    @Override
    public void newNhCell(Long value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.NORMOHOUR, props));
            this.cell.setCellValue((double)value.longValue() / 3600000.0);
        }
    }

    @Override
    public boolean canWriteStatistics() {
        return true;
    }

    @Override
    public void addStatisticsFormula(TableWriter.ColumnStatisticsType type) {
        CellStyle cellStyle = this.book.createCellStyle();
        cellStyle.cloneStyleFrom(this.cell.getCellStyle());
        cellStyle.setBorderTop(BorderStyle.DOUBLE);
        this.cell.setCellStyle(cellStyle);
        if (type == TableWriter.ColumnStatisticsType.NONE) {
            return;
        }
        this.cell.setCellFormula(this.createCellFormula(type));
    }

    private String createCellFormula(TableWriter.ColumnStatisticsType type) {
        String function = switch (type) {
            default -> throw new IncompatibleClassChangeError();
            case TableWriter.ColumnStatisticsType.NONE -> throw new IllegalStateException();
            case TableWriter.ColumnStatisticsType.SUM -> "SUM";
            case TableWriter.ColumnStatisticsType.AVG -> "AVERAGE";
            case TableWriter.ColumnStatisticsType.MIN -> "MIN";
            case TableWriter.ColumnStatisticsType.MAX -> "MAX";
        };
        String rowTo = String.valueOf(this.rowNo - 1);
        String columnName = ExcelTableWriter.getExcelGeneratedColumnName(this.cellNo - 1);
        return function + "(" + columnName + "2:" + columnName + rowTo + ")";
    }

    @Nonnull
    public static String getExcelGeneratedColumnName(int zeroBasedCellNo) {
        return zeroBasedCellNo < 0 ? "" : ExcelTableWriter.getExcelGeneratedColumnName(zeroBasedCellNo / 26 - 1) + (char)(zeroBasedCellNo % 26 + 65);
    }

    @Override
    public void newIntegerCell(Long value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.INTEGER, props));
            this.cell.setCellValue((double)value.longValue());
        }
    }

    @Override
    public void newFloatCell(Double value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.FLOAT, props));
            this.cell.setCellValue(value.doubleValue());
        }
    }

    @Override
    public void newPercentCell(Double value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            this.newCell(this.getCellStyle(CellStyleInfo.Type.PERCENT, props));
            this.cell.setCellValue(value.doubleValue());
        }
    }

    @Override
    public void newColorCell(Color value, TableWriter.CellProperties props) {
        if (value == null) {
            this.newEmptyCell(props);
        } else {
            Workbook workbook = this.book;
            if (workbook instanceof SXSSFWorkbook) {
                SXSSFWorkbook b = (SXSSFWorkbook)workbook;
                CellStyle xcs = b.createCellStyle();
                xcs.setFillPattern(FillPatternType.SOLID_FOREGROUND);
                IndexedColorMap indexedColors = b.getXSSFWorkbook().getStylesSource().getIndexedColors();
                xcs.setFillForegroundColor(new XSSFColor(value, indexedColors).getIndexed());
                this.newCell(xcs);
                this.cell.setCellStyle(xcs);
            } else {
                this.newStringCell("#" + Integer.toHexString(value.getRGB()), props);
            }
        }
    }

    @Override
    public boolean addMultipleLinks(@Nonnull List<TableWriter.CellLink> links) {
        return false;
    }

    @Override
    public void addLink(@Nonnull String absoluteUrl, @Nullable TableWriter.CellProperties props) {
        if (this.linksOnSheet >= 65530) {
            if (this.linksOnSheet == 65530) {
                log.info("The number of hyperlinks on sheet {} has reached its limit. Some links will be missing.", (Object)this.sheet.getSheetName());
            }
            ++this.linksOnSheet;
            return;
        }
        try {
            Hyperlink link = this.creationHelper.createHyperlink(HyperlinkType.URL);
            link.setAddress(absoluteUrl);
            this.cell.setHyperlink(link);
            ++this.linksOnSheet;
            this.cell.setCellStyle(this.getCellStyle(CellStyleInfo.Type.HYPERLINK, props));
        }
        catch (Exception e) {
            this.addError("Error adding link", "Failed add URL:'" + absoluteUrl + "', " + e.getMessage());
        }
    }

    @Override
    public void addDescription(String text, boolean useMonospaceFont) {
        if (Strings.isNullOrEmpty(text)) {
            return;
        }
        if (this.commentCount > 10000) {
            return;
        }
        ++this.commentCount;
        text = ReportUtilities.stripHtml(text);
        Comment comment = this.createComment(text, useMonospaceFont);
        this.cell.setCellComment(comment);
    }

    @Nonnull
    private Comment createComment(String text, boolean useMonospaceFont) {
        ClientAnchor anchor = this.createAnchor(text);
        Comment comment = this.drawingPatriarch.createCellComment(anchor);
        RichTextString richText = this.creationHelper.createRichTextString(text);
        if (useMonospaceFont) {
            richText.applyFont(this.monospaceFont);
        }
        comment.setString(richText);
        comment.setVisible(false);
        comment.setAuthor("");
        comment.setRow(this.cell.getRowIndex());
        comment.setColumn(this.cell.getColumnIndex());
        return comment;
    }

    @Nonnull
    private ClientAnchor createAnchor(String text) {
        int lines = ExcelTableWriter.countLines(text);
        ClientAnchor anchor = this.creationHelper.createClientAnchor();
        anchor.setCol1(this.cell.getColumnIndex());
        anchor.setCol2(this.cell.getColumnIndex() + 5);
        anchor.setRow1(this.row.getRowNum());
        anchor.setRow2(this.row.getRowNum() + lines);
        return anchor;
    }

    private static int countLines(String text) {
        int lines = 1;
        for (int i = 0; i < text.length(); ++i) {
            if (text.charAt(i) != '\n') continue;
            ++lines;
        }
        return lines;
    }

    @Override
    public void newEmptyCell(TableWriter.CellProperties props) {
        this.newCell(props == null ? null : this.getCellStyle(CellStyleInfo.Type.TEXT, props));
    }

    private void autosizeCells() {
        if (this.autosized) {
            return;
        }
        for (int i = 0; i <= this.maxCellNo; ++i) {
            this.sheet.autoSizeColumn(i, false);
        }
        if (this.sheet instanceof SXSSFSheet) {
            ((SXSSFSheet)this.sheet).untrackAllColumnsForAutoSizing();
        }
        this.autosized = true;
    }

    @Override
    public void closeWriter() {
        log.info(String.format("Writing Excel [%d x %d]...", this.cellNo, this.rowNo));
        this.autosizeCells();
        try {
            this.book.write(this.output);
            this.output.flush();
            if (this.book instanceof SXSSFWorkbook && !((SXSSFWorkbook)this.book).dispose()) {
                log.warn("Error deleting temporary files.");
            }
            log.info("Done writing Excel.");
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (this.shouldClose) {
                try {
                    this.output.close();
                }
                catch (IOException iOException) {}
            }
        }
    }

    private static class CellStyleInfo {
        String bckgColor;
        String txtColor;
        Type type;

        CellStyleInfo(Type type, String bckgColor, String txtColor) {
            this.bckgColor = bckgColor;
            this.txtColor = txtColor;
            this.type = type;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof CellStyleInfo)) {
                return false;
            }
            CellStyleInfo that = (CellStyleInfo)o;
            return Objects.equals(this.bckgColor, that.bckgColor) && Objects.equals(this.txtColor, that.txtColor) && this.type == that.type;
        }

        public int hashCode() {
            int result = this.bckgColor != null ? this.bckgColor.hashCode() : 0;
            result = 31 * result + (this.txtColor != null ? this.txtColor.hashCode() : 0);
            result = 31 * result + this.type.hashCode();
            return result;
        }

        public static enum Type {
            TEXT,
            FLOAT,
            DATETIME,
            DATE,
            TIME,
            DURATION_HOUR,
            DURATION_DAY,
            NORMOHOUR,
            PERCENT,
            HEADER,
            HYPERLINK,
            INTEGER;

        }
    }
}

